<#
================================================================================
  Sprite Sheet Generator Script
  Created for UnityKing.com
  License: UnityKing Free Public License (UFPL)

  This script is released as free, open-use software for the community.
  You are allowed to:
      - Use it for personal and commercial projects
      - Modify or extend it
      - Redistribute it with attribution

  Attribution Requirements:
      Include a visible reference to UnityKing.com in the script header
      and wherever redistribution occurs.

  Disclaimer:
      This software is provided "as-is" without any warranty of any kind.
      The author and UnityKing.com are not responsible for any damage
      or data loss resulting from the use of this script.

  Website: https://UnityKing.com
================================================================================
#>

Add-Type -AssemblyName System.Drawing

Write-Host "=== SPRITE SHEET GENERATOR ==="
$folder = Read-Host "Enter images folder path"
$cols = [int](Read-Host "Number of columns (e.g., 5)")
$rows = [int](Read-Host "Number of rows (e.g., 2)")
$hSpace = [int](Read-Host "Horizontal spacing between images (px)"
)
$vSpace = [int](Read-Host "Vertical spacing between images (px)")
$padLeft = [int](Read-Host "Left padding (px)")
$padRight = [int](Read-Host "Right padding (px)")
$padTop = [int](Read-Host "Top padding (px)")
$padBottom = [int](Read-Host "Bottom padding (px)")
$outputFolder = "$folder\SpriteSheets"

if (!(Test-Path $outputFolder)) {
    New-Item -ItemType Directory -Path $outputFolder | Out-Null
}

$images = Get-ChildItem $folder -Include *.png, *.jpg, *.jpeg -Recurse
$imgCount = $images.Count

if ($imgCount -eq 0) {
    Write-Host "No images found. Exiting..."
    exit
}

Write-Host "`nFound $imgCount images."

# Load first image to get universal size
$sample = [System.Drawing.Image]::FromFile($images[0].FullName)
$imgW = $sample.Width
$imgH = $sample.Height

$maxPerSheet = $cols * $rows
$sheetIndex = 1
$start = 0

while ($start -lt $imgCount) {

    $end = [Math]::Min($start + $maxPerSheet - 1, $imgCount - 1)
    $currentImages = $images[$start..$end]

    # Calculate actual rows needed (for last sheet)
    $needed = $currentImages.Count
    $actualRows = [Math]::Ceiling($needed / $cols)

    # Calculate sheet size
    $sheetW = ($imgW * $cols) + ($hSpace * ($cols - 1)) + $padLeft + $padRight
    $sheetH = ($imgH * $actualRows) + ($vSpace * ($actualRows - 1)) + $padTop + $padBottom

    $sheet = New-Object System.Drawing.Bitmap($sheetW, $sheetH)
    $gfx = [System.Drawing.Graphics]::FromImage($sheet)
    $gfx.Clear([System.Drawing.Color]::Transparent)

    $i = 0
    foreach ($imgFile in $currentImages) {
        $bmp = [System.Drawing.Image]::FromFile($imgFile.FullName)

        $col = $i % $cols
        $row = [Math]::Floor($i / $cols)

        $x = $padLeft + ($col * ($imgW + $hSpace))
        $y = $padTop + ($row * ($imgH + $vSpace))

        $gfx.DrawImage($bmp, $x, $y)

        $bmp.Dispose()
        $i++
    }

    $outFile = "$outputFolder\SpriteSheet_$sheetIndex.png"
    $sheet.Save($outFile, [System.Drawing.Imaging.ImageFormat]::Png)

    Write-Host "Generated: SpriteSheet_$sheetIndex.png"

    $gfx.Dispose()
    $sheet.Dispose()

    $sheetIndex++
    $start += $maxPerSheet
}

Write-Host "`nAll sprite sheets generated in: $outputFolder"